#!/usr/bin/env lua
-- (prepared for Unix)

-- Process a file with data description as Lua code

require "auxiliar"
require "datatest"
require "template"

local fname = arg[1]      -- filename to process
local ERR = false         -- a pseudo-global error
local process = {}        -- to store element processes

--[[
   Box processing (an example)
   
   The routine has three parts.
      * a testing by comparing the element with template,
      * a post-testing to possibly catch a difficult kind of error,
      * a processing of the element, completed with default values
        if needed.
--]]
function process.box (e)
  if ERR then return end   -- if other element has produced an error

  message("-- Testing " .. (e.id or "a box"))

  message(" ++ input data\n")
  message(table.show(e))

  -- testing table e with template.box
  local ok, msg = datatest.main(template.box, e, "box")
  if msg ~= "" then message(msg) end
  if not ok then
    --message("-- Error in " .. (e.id or "a box") .."\n")
    ERR = true
    return
  end

  --[[
      This part is for trapping errors like
          id , "box2",    instead of    id = "box2",
      when id is and optional field and not _STRICT
  --]]
  if #e > 0 then
    local t = ""
    for i, v in pairs(e) do 
      if type(i) == "number" then t = t .. tostring(v) end
    end
    message("\n-- Warning. Probable error in " .. (e.id or "a box")
            .. " about: " .. t .. "\n")
  end

  -- processing table e (completed with default values if exist)
  message("-- Processing " .. (e.id or "a box"))
  message(" ++ actual data\n")
  message(table.show(e))
end

-- arc processing
-- this is nearly a copy of process.box
-- in productive code they will obviously differ
function process.arc (e)
  if ERR then return end
  message("-- Testing " .. (e.id or "an arc"))
  message(" ++ input data\n")
  message(table.show(e))
  local ok, msg = datatest.main(template.arc, e, "arc")
  if msg ~= "" then message(msg) end
  if not ok then
    --message("-- Error in " .. (e.id or "an arc") .."\n")
    ERR = true
    return
  end
  if #e > 0 then
    local t = ""
    for i, v in pairs(e) do 
      if type(i) == "number" then t = t .. tostring(v) end
    end
    message("\n-- Warning. Probable error in " .. (e.id or "an arc") 
            .. " about: " .. t .. "\n")
  end
  message("-- Processing " .. (e.id or "an arc"))
  message(" ++ actual data\n")
  message(table.show(e))
end

--============================================================

local function extractinfo(fname, msg)
  local errmatch = "([^:]+):(%d+):"
  local place, line = string.match(msg, errmatch)
  if place == fname then
    -- error in the file in process
    -- NOTE: Lua error messages are not very good for the user
    --       in some cases.
    return string.format("***** Error in %s, in line %d or before\n",
                         place, line)  -- .."\n"..msg  <-- for debugging
  else
    -- error in some part of the program
    return "####   " .. msg .. "\n"
  end
end

-- controlled loadfile
local function myloadfile(fname)
  local main, msg = loadfile(fname)
  if main then return main end
  message(extractinfo(fname, msg))
  os.exit(1)
end

-- controlled pcall
local function mypcall(fname, fun)
  setfenv(fun, {math = math, string = string, table = table,
                box = process.box, arc = process.arc})
  local ok, msg = pcall(fun)
  if not ok then
    message(extractinfo(fname, msg))
    os.exit(2)
  end
end

--============================================================
-- main program

message("\n--" .. string.rep("=", 60) .."\n")
message("-- File loading ...\n")

-- syntactical errors in fname are catched here
local main = myloadfile(fname)

--box = process.box
--arc = process.arc

message("--" .. string.rep("=", 60) .."\n")
message("-- File testing and processing ...\n")

-- other possible errors in fname processing are catched here
mypcall(fname, main)

if ERR then os.exit(3) end  -- a last resort

message("--" .. string.rep("=", 60) .."\n")

-- all ok

message("--> DONE\n")
